/* --------------------------------------------------------------
 initialize.js 2015-10-13 gm
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2015 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

// Initialize base engine object. Every other part of the engine will refer to this
// central object for the core operations.

window.jse = {
	core: {},
	libs: {},
	constructors: {}
};

/**
 * ## JS Engine Initialization
 *
 * The document-ready event of the page will trigger the JavaScript Engine initialization. The
 * Engine requires a global configuration object "window.JSEngineConfiguration" to be pre-defined
 * in order to retrieve the basic configuration info. After a successful initialization the
 * EngineConfiguration object is removed from the global scope.
 *
 * Important: Place this file at the top of the concatenated files that will be included in the "jse.js".
 *
 * ## Configuration Sample
 *
 * ```javascript
 * window.JSEngineConfiguration = {
 *     environment: 'production',
 *     shopUrl: 'http://www.shop.de',
 *     translations: {
 *         'sectionName': { 'translationKey': 'translationValue' },
 *         'anotherSection': { ... }
 *     },
 *     languageCode: 'de',
 *     pageToken: '9asd7f9879sd8f79s98s7d98f'
 * };
 * ```
 *
 * @namespace JSE/Core/initialize
 */

$(document).ready(function () {

	'use strict';

	try {
		// Check if global JSEngineConfiguration object is defined.
		if (typeof window.JSEngineConfiguration === 'undefined') {
			throw new Error(
				'The "window.JSEngineConfiguration" object is not defined in the global scope. This ' +
				'object is required by the JSEngine upon its initialization. Check "core/initialize" ' +
				'documentation page.');
		}

		// Parse JSEngineConfiguration object.
		jse.core.config.init(window.JSEngineConfiguration);

		// Initialize engine module collections.
		window.engineStartTime = new Date().getTime();
		jse.core.engine.init([
			new jse.constructors.Collection('extensions', 'extension'),
			new jse.constructors.Collection('controllers', 'controller'),
			new jse.constructors.Collection('widgets', 'widget'),
			new jse.constructors.Collection('compatibility', 'compatibility')
		]);
	} catch (exception) {
		jse.core.debug.error('Unexpected error during JS Engine initialization!', exception);
	}
});

/* --------------------------------------------------------------
 collection.js 2015-10-13 gm
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2015 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

(function() {

	'use strict';

	/**
	 * Class Collection
	 *
	 * @param {string} name The collection name - must be unique.
	 * @param {string} attribute The attribute that will trigger collection's modules.
	 * @param {object} namespace (optional) The namespace instance where the collection belongs.
	 *
	 * @constructor JSE/Core/Collection
	 */
	function Collection(name, attribute, namespace) {
		this.name = name;
		this.attribute = attribute;
		this.namespace = namespace;
		this.cache = {
			modules: {},
			data: {}
		};
	}

	/**
	 * Define a new engine module.
	 *
	 * This function will define a new module into the engine. Each extension will
	 * be stored in the collection's cache to prevent unnecessary file transfers via RequireJS.
	 * The same happens with the default configuration that append to the module definition.
	 *
	 * @param {string} name Name of the module.
	 * @param {array} dependencies Array of libraries that this module depends on (will be loaded asynchronously
	 *                  & apply only file names without extension e.g. ["emails"]).
	 * @param {object} code Contains the module code (function).
	 *
	 * @name core/collection.module
	 * @public
	 * @method
	 */
	Collection.prototype.module = function(name, dependencies, code) {
		// Collection instance alias.
		var collection = this;

		// Check if required values are available and of correct type.
		if (!name || typeof name !== 'string' || typeof code !== 'function') {
			jse.core.debug.warn('Registration of the module failed, due to bad function call', arguments);
			return false;
		}

		// Check if the module is already defined.
		if (collection.cache.modules[name]) {
			jse.core.debug.warn('Registration of module "' + name + '" skipped, because it already exists.');
			return false;
		}

		// Store the module to cache so that it can be used later.
		collection.cache.modules[name] = {
			code: code,
			dependencies: dependencies
		};
	};

	/**
	 * [DEPRECATED] Register a module definition
	 *
	 * This method exists only for fallback support. It will be removed in the future so please
	 * use the "module" function above.
	 *
	 * @todo Remove the fallback in the next engine version.
	 *
	 * @deprecated since version 1.2.0
	 */
	Collection.prototype.register = function(name, version, dependencies, code, defaults) {
		jse.core.debug.warn('The Collection.prototype.register() method is deprecated as of v1.2.0, use the '
		                    + 'Collection.prototype.module() method instead -- ' + name);
		this.module(name, dependencies, code);
	};

	/**
	 * Initialize Module Collection
	 *
	 * This method will trigger the page modules initialization. It will search all
	 * the DOM for the "data-gx-extension", "data-gx-controller" or
	 * "data-gx-widget" attributes and load the relevant scripts through RequireJS.
	 *
	 * @param {object} $parent Parent element will be used to search for the required modules.
	 * @param {object} namespaceDeferred Deferred object that gets processed after the module initialization is finished.
	 *
	 * @name core/engine.init
	 * @public
	 * @method
	 */
	Collection.prototype.init = function($parent, namespaceDeferred) {
		// Collection instance alias.
		var collection = this;

		// Store the namespaces reference of the collection.
		if (!collection.namespace) {
			throw new Error('Collection cannot be initialized without its parent namespace instance.');
		}

		// Set the default parent-object if none was given.
		if (typeof $parent === 'undefined' || $parent === null) {
			$parent = $('html');
		}

		var attribute = 'data-' + collection.namespace.name + '-' + collection.attribute,
			deferredCollection = [];

		$parent
			.filter('[' + attribute + ']')
			.add($parent.find('[' + attribute + ']'))
			.each(function() {
				var $element = $(this),
					modules = $element.attr(attribute);

				$element.removeAttr(attribute);
				$.each(modules.trim().split(' '), function(index, name) {
					var deferred = $.Deferred();
					deferredCollection.push(deferred);

					jse.core.module_loader
					   .load($element, name, collection)
					   .done(function(module) {
						   module.init(deferred);
					   })
					   .fail(function(error) {
						   deferred.reject();
						   // Log the error in the console but do not stop the engine execution.
						   jse.core.debug.error('Could not load module: ' + name, error);
					   });
				});
			});

		// If an namespaceDeferred is given resolve or reject it depending on the module initialization status.
		if (namespaceDeferred) {
			if (deferredCollection.length === 0 && namespaceDeferred) {
				namespaceDeferred.resolve();
			}

			$.when.apply(undefined, deferredCollection).promise()
			 .done(function() {
				 namespaceDeferred.resolve();
			 })
			 .fail(function() {
				 namespaceDeferred.fail();
			 });
		}
	};

	jse.constructors.Collection = Collection;
})();

/* --------------------------------------------------------------
 module.js 2016-02-04
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 ----------------------------------------------------------------
 */

(function() {

	'use strict';

	/**
	 * Class Module
	 *
	 * @param {object} $element Module element selector object.
	 * @param {string} name The module name (might contain the path)
	 * @param {object} collection The collection instance of the module.
	 *
	 * @constructor JSE/Core/Module
	 */
	function Module($element, name, collection) {
		this.$element = $element;
		this.name = name;
		this.collection = collection;
	}

	/**
	 * Initialize the module execution.
	 *
	 * This function will execute the "init" method of each module.
	 *
	 * @param {object} collectionDeferred Deferred object that gets processed after the module initialization is finished
	 *
	 * @public
	 */
	Module.prototype.init = function(collectionDeferred) {
		// Store module instance alias.
		var module = this,
			cached = module.collection.cache.modules[module.name],
			promise = collectionDeferred.promise(),
			watchdog = null;

		try {
			if (!cached) {
				throw new Error('Module "' + module.name + '" could not be found in the collection cache.');
			}

			var data = _getModuleData(module),
				instance = cached.code.call(module.$element, data);

			// Provide a done function that needs to be called from the module, in order to inform that the module 
			// "init" function was completed successfully.
			var done = function() {
				module.$element.trigger('module.initialized', [
					{
						module: module.name
					}
				]);
				jse.core.debug.info('Module "' + module.name + '" initialized successfully.');
				collectionDeferred.resolve();
				clearTimeout(watchdog);
			};

			// Fallback support for the _initFinished function.
			// @todo Remove the fallback in the next engine version.
			instance._initFinished = function() {
				jse.core.debug.warn('The "_initFinished" function is deprecated as of v1.2.0, use the new '
				                    + 'module initialization instead -- ' + module.name);
				done();
			};

			// Fallback support for the _data function.
			// @todo Remove the fallback in the next engine version.
			instance._data = function($element) {
				jse.core.debug.warn('The "_data" function is deprecated as of v1.2.0, use jQuery data() '
				                    + 'function instead -- ' + module.name);

				var initialData = $element.data(),
					filteredData = {};

				// Searches for module relevant data inside the main-data-object.
				// Data for other widgets will not get passed to this widget
				$.each(initialData, function(key, value) {
					if (key.indexOf(module.name) === 0 || key.indexOf(module.name.toLowerCase()) === 0) {
						var newKey = key.substr(module.name.length);
						newKey = newKey.substr(0, 1).toLowerCase() + newKey.substr(1);
						filteredData[newKey] = value;
					}
				});

				return filteredData;
			};

			// Load the module data before the module is loaded.
			_loadModuleData(instance)
				.done(function() {
					_syncLibsFallback();
					
					// Reject the collectionDeferred if the module isn't initialized after 15 seconds.
					watchdog = setTimeout(function() {
						jse.core.debug.warn('Module was not initialized after 15 seconds! -- ' + module.name);
						collectionDeferred.reject();
					}, 15000);
					
					instance.init(done);
				})
				.fail(function(error) {
					collectionDeferred.reject();
					jse.core.debug.error('Could not load module\'s meta data.', error);
				});
		} catch (exception) {
			collectionDeferred.reject();
			jse.core.debug.error('Cannot initialize module "' + module.name + '"', exception);
		}

		return promise;
	};

	/**
	 * Parse the module data attributes.
	 *
	 * @param {object} module The module instance to be parsed.
	 *
	 * @returns {object} Returns an object that contains the data of the module.
	 *
	 * @private
	 */
	var _getModuleData = function(module) {
		var data = {};

		$.each(module.$element.data(), function(name, value) {
			if (name.indexOf(module.name) === 0 || name.indexOf(module.name.toLowerCase()) === 0) {
				var key = name.substr(module.name.length);
				key = key.substr(0, 1).toLowerCase() + key.substr(1);
				data[key] = value;
				module.$element.removeAttr('data-' + module.name + '-' + key);
			}
		});

		return data;
	};

	/**
	 * Modules return objects which might contain requirements.
	 *
	 * @param {object} instance Module instance object.
	 *
	 * @return {object} Returns a promise object that will be resolved when the data are fetched.
	 *
	 * @private
	 */
	var _loadModuleData = function(instance) {
		var deferred = $.Deferred();

		try {
			var promises = [];

			if (instance.model) {
				$.each(instance.model, function(index, url) {
					var modelDeferred = $.Deferred();
					promises.push(modelDeferred);
					$.getJSON(url)
					 .done(function(response) {
						 instance.model[index] = response;
						 modelDeferred.resolve(response);
					 })
					 .fail(function(error) {
						 modelDeferred.reject(error);
					 });
				});
			}

			if (instance.view) {
				$.each(instance.view, function(index, url) {
					var viewDeferred = $.Deferred();
					promises.push(viewDeferred);
					$.get(url)
					 .done(function(response) {
						 instance.view[index] = response;
						 viewDeferred.resolve(response);
					 })
					 .fail(function(error) {
						 viewDeferred.reject(error);
					 });
				});
			}

			$.when
			 .apply(undefined, promises)
			 .promise()
			 .done(function() {
				 deferred.resolve();
			 })
			 .fail(function(error) {
				 deferred.reject(new Error('Cannot load data for module "' + module.name + '".', error));
			 });
		} catch (exception) {
			deferred.resolve(exception);
		}

		return deferred.promise();
	};

	/**
	 * Engine Libs fallback definition.
	 *
	 * Old modules use the libs under the window.gx.libs object. This method will make sure
	 * that this object is synchronized with the jse.libs until every old module definition is
	 * updated.
	 *
	 * @todo Remove the fallback in the next engine version.
	 *
	 * @private
	 */
	var _syncLibsFallback = function() {
		if (typeof window.gx === 'undefined') {
			window.gx = {};
		}
		window.gx.libs = jse.libs;
	};

	jse.constructors.Module = Module;
})();

/* --------------------------------------------------------------
 namespace.js 2015-10-13 gm
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2015 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

(function () {

	'use strict';

	/**
	 * Class Namespace
	 *
	 * @param {string} name The namespace name must be unique within the app.
	 * @param {string} source Complete URL to the namespace modules directory (without trailing slash).
	 * @param {array} collections Contains collection instances to be included in the namespace.
	 *
	 * @constructor JSE/Core/Namespace
	 */
	function Namespace(name, source, collections) {
		this.name = name;
		this.source = source;
		this.collections = collections; // contains the default instances
	}

	/**
	 * Initialize the namespace collections.
	 *
	 * This method will create new collection instances based in the original ones.
	 */
	Namespace.prototype.init = function () {
		var deferredCollection = [];

		for (var index in this.collections) {
			var collection = this.collections[index],
			    deferred = $.Deferred();

			deferredCollection.push(deferred);
			
			this[collection.name] = new jse.constructors.Collection(collection.name, collection.attribute, this);
			this[collection.name].init(null, deferred);
		}

		if (deferredCollection.length === 0) {
			return $.Deferred().resolve();
		}

		return $.when.apply(undefined, deferredCollection).promise();

	};

	jse.constructors.Namespace = Namespace;
})();

/* --------------------------------------------------------------
 about.js 2015-10-14 gm
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2015 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Get information about the JS Engine.
 *
 * Execute the `jse.about()` command and you will get a new log entry in the
 * console with info about the engine. The "about" method is only available in
 * the "development" environment of the engine.
 *
 * @namespace JSE/Core/about
 */
$(document).ready(function() {

	'use strict';

	if (jse.core.config.get('environment') === 'production') {
		return;
	}

	jse.about = function () {
		var info = [
			'JS ENGINE v' + jse.core.config.get('version') + ' © GAMBIO GMBH',
			'----------------------------------------------------------------',
			'The JS Engine enables developers to load automatically small pieces of javascript code by',
			'placing specific data attributes to the HTML markup of a page. It was built with modularity',
			'in mind so that modules can be reused into multiple places without extra effort. The engine',
			'contains namespaces which contain collections of modules, each one of whom serve a different',
			'generic purpose.',
			'',
			'Visit http://developers.gambio.de for complete reference of the JS Engine.',
			'',
			'FALLBACK INFORMATION',
			'----------------------------------------------------------------',
			'Since the engine code becomes bigger there are sections that need to be refactored in order',
			'to become more flexible. In most cases a warning log will be displayed at the browser\'s console',
			'whenever there is a use of a deprecated function. Below there is a quick list of fallback support',
			'that will be removed in the future versions of the engine.',
			'',
			'1. The main engine object was renamed from "gx" to "jse" which stands for the JavaScript Engine.',
			'2. The "gx.lib" object is removed after a long deprecation period. You should update the modules ',
			'   that contained calls to the functions of this object.',
			'3. The gx.<collection-name>.register function is deprecated by v1.2, use the ',
			'   <namespace>.<collection>.module() instead.'
		];

		jse.core.debug.info(info.join('\n'));
	};

});

/* --------------------------------------------------------------
 config.js 2015-10-20 gm
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2015 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

jse.core.config = jse.core.config || {};

/**
 * JS Engine Configuration Object
 *
 * Once the config object is initialized you cannot change its values. This is done in order to
 * prevent unpleasant situations where one code section changes a core config setting that affects
 * another code section in a way that is hard to discover.
 *
 * ```javascript
 * var shopUrl = jse.core.config.get('shopUrl');
 * ```
 *
 * @namespace JSE/Core/config
 */
(function (exports) {

	'use strict';

	// ------------------------------------------------------------------------
	// CONFIGURATION VALUES
	// ------------------------------------------------------------------------

	var config = {
		/**
		 * Engine Version
		 *
		 * @type {string}
		 */
		version: '1.2.0',

		/**
		 * Shop URL
		 *
		 * e.g. 'http://shop.de
		 *
		 * @type {string}
		 */
		shopUrl: null,

		/**
		 * URL to JSEngine Directory.
		 *
		 * e.g. 'http://shop.de/JSEngine
		 *
		 * @type {string}
		 */
		engineUrl: null,

		/**
		 * Engine Environment
		 *
		 * Defines the functionality of the engine in many sections.
		 *
		 * @type {string}
		 */
		environment: 'production',

		/**
		 * HTML Attribute Prefix
		 *
		 * This will prefix the HTML attributes that have a special meaning to the JSEngine. Should
		 * be mostly a short string.
		 *
		 * @type {string}
		 */
		prefix: 'gx',

		/**
		 * Translations Object
		 *
		 * Contains the loaded translations to be used within JSEngine.
		 *
		 * @see gx.libs.lang object
		 * @type {object}
		 */
		translations: {},

		/**
		 * Current Language Code
		 *
		 * @type {string}
		 */
		languageCode: '',

		/**
		 * Set the debug level to one of the following: 'DEBUG', 'INFO', 'LOG', 'WARN', 'ERROR', 'ALERT', 'SILENT'
		 *
		 * @type {string}
		 */
		debug: 'SILENT',

		/**
		 * Use cache busting technique when loading modules.
		 *
		 * @see jse.core.debug object
		 * @type {bool}
		 */
		cacheBust: true,

		/**
		 * Load minified files.
		 *
		 * @type {bool}
		 */
		minified: true,

		/**
		 * Whether the client has a mobile interface.
		 *
		 * @type {bool}
		 */
		mobile: false,

		/**
		 * Whether the client supports touch events.
		 *
		 * @type {bool}
		 */
		touch: (window.ontouchstart || window.onmsgesturechange) ? true : false,

		/**
		 * Specify the path for the file manager.
		 *
		 * @type {string}
		 */
		filemanager: 'includes/ckeditor/filemanager/index.html',

		/**
		 * Page token to include in every AJAX request.
		 *
		 * The page token is used to avoid CSRF attacks. It must be provided by the
		 * backend and it will be validated there.
		 *
		 * @type {string}
		 */
		pageToken: '',

		/**
		 * Defines whether the history object is available.
		 */
		history: history && history.replaceState && history.pushState
	};

	// ------------------------------------------------------------------------
	// PUBLIC METHODS
	// ------------------------------------------------------------------------

	/**
	 * Get a configuration value.
	 *
	 * @param {string} name The configuration value name to be retrieved.
	 *
	 * @returns {*} Returns the config value.
	 *
	 * @name core/config.init
	 * @public
	 * @method
	 */
	exports.get = function (name) {
		return config[name];
	};

	/**
	 * Initialize the JS Engine config object.
	 *
	 * This method will parse the global "JSEngineConfiguration" object and then remove
	 * it from the global scope so that it becomes the only config source for javascript.
	 *
	 * Notice: The only required JSEngineConfiguration values are the "environment" and the "shopUrl".
	 *
	 * @param {object} jsEngineConfiguration Must contain information that define core operations
	 * of the engine. Check the "libs/initialize" entry of the engine documentation.
	 *
	 * @name core/config.init
	 * @public
	 * @method
	 */
	exports.init = function (jsEngineConfiguration) {
		config.environment = jsEngineConfiguration.environment;
		config.shopUrl = jsEngineConfiguration.shopUrl;

		if (config.environment === 'development') {
			config.cacheBust = false;
			config.minified = false;
			config.debug = 'DEBUG';
		}

		if (typeof jsEngineConfiguration.engineUrl !== 'undefined') {
			config.engineUrl = jsEngineConfiguration.engineUrl;
		} else {
			config.engineUrl = config.shopUrl + '/JSEngine/build';
		}

		if (typeof jsEngineConfiguration.translations !== 'undefined') {
			config.translations = jsEngineConfiguration.translations;

			$.each(config.translations, function (sectionName, sectionTranslations) {
				jse.core.lang.addSection(sectionName, sectionTranslations);
			});
		}

		if (typeof jsEngineConfiguration.prefix !== 'undefined') {
			config.prefix = jsEngineConfiguration.prefix;
		}

		if (typeof jsEngineConfiguration.languageCode !== 'undefined') {
			config.languageCode = jsEngineConfiguration.languageCode;
		}

		if (typeof jsEngineConfiguration.pageToken !== 'undefined') {
			config.pageToken = jsEngineConfiguration.pageToken;
		}

		// Initialize the module loader object.
		jse.core.module_loader.init();

		// Destroy global EngineConfiguration object.
		delete window.JSEngineConfiguration;
	};

}(jse.core.config));

/* --------------------------------------------------------------
 debug.js 2015-10-13 gm
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2015 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

jse.core.debug = jse.core.debug || {};

/**
 * JS Engine debug object.
 *
 * This object provides an wrapper to the console.log function and enables easy use
 * of the different log types like "info", "warning", "error" etc.
 *
 * @namespace JSE/Core/debug
 */
(function (/** @lends JSE/Core/debug */ exports) {
	'use strict';

	// ------------------------------------------------------------------------
	// VARIABLE DEFINITION
	// ------------------------------------------------------------------------

	/**
	 * All possible debug levels in the order of importance.
	 *
	 * @name Core/Debug.levels
	 * @public
	 * @type {array}
	 */
	var levels = ['DEBUG', 'INFO', 'LOG', 'WARN', 'ERROR', 'ALERT', 'SILENT'];

	/**
	 * Executes the correct console/alert statement.
	 *
	 * @name Core/Debug._execute
	 * @private
	 * @method
	 *
	 * @param {object} caller (optional) Contains the caller information to be displayed.
	 * @param {object} data (optional) Contains any additional data to be included in the debug output.
	 */
	var _execute = function (caller, data) {
		var currentLogIndex = levels.indexOf(caller),
			allowedLogIndex = levels.indexOf(jse.core.config.get('debug')),
			consoleMethod = null;

		if (currentLogIndex >= allowedLogIndex) {
			consoleMethod = caller.toLowerCase();

			if (consoleMethod === 'alert') {
				alert(JSON.stringify(data));
				return;
			}

			if (consoleMethod === 'mobile') {
				var $dbgLayer = $('.mobileDbgLayer');
				if (!$dbgLayer.length) {
					$dbgLayer = $('<div />');
					$dbgLayer.addClass('mobileDbgLayer');
					$dbgLayer.css({
						'position': 'fixed',
						'top': 0,
						'left': 0,
						'max-height': '50%',
						'min-width': '200px',
						'max-width': '300px',
						'background-color': 'crimson',
						'z-index': 100000,
						'overflow': 'scroll'
					});
					$('body').append($dbgLayer);
				}

				$dbgLayer.append('<p>' + JSON.stringify(data) + '</p>');
				return;
			}

			if (typeof console === 'undefined') {
				return; // There is no console support so do not proceed.
			}

			if (typeof console[consoleMethod].apply === 'function' || typeof console.log.apply === 'function') {
				if (typeof console[consoleMethod] !== 'undefined') {
					console[consoleMethod].apply(console, data);
				} else {
					console.log.apply(console, data);
				}
			} else {
				console.log(data);
			}

			return true;
		}
		return false;
	};

	// ------------------------------------------------------------------------
	// VARIABLE EXPORT
	// ------------------------------------------------------------------------

	/**
	 * Replaces console.debug
	 *
	 * @params {all} Any data that should be shown in the console statement
	 *
	 * @name Core/Debug.debug
	 * @public
	 * @method
	 */
	exports.debug = function () {
		_execute('DEBUG', arguments);
	};

	/**
	 * Replaces console.info
	 *
	 * @params {all} Any data that should be shown in the console statement
	 *
	 * @name Core/Debug.info
	 * @public
	 * @method
	 */
	exports.info = function () {
		_execute('INFO', arguments);
	};

	/**
	 * Replaces console.log
	 *
	 * @params {all} Any data that should be shown in the console statement
	 *
	 * @name Core/Debug.log
	 * @public
	 * @method
	 */
	exports.log = function () {
		_execute('LOG', arguments);
	};

	/**
	 * Replaces console.warn
	 *
	 * @params {all} Any data that should be shown in the console statement
	 *
	 * @name Core/Debug.warn
	 * @public
	 * @method
	 */
	exports.warn = function () {
		_execute('WARN', arguments);
	};

	/**
	 * Replaces console.error
	 *
	 * @param {all} Any data that should be shown in the console statement
	 *
	 * @name Core/Debug.error
	 * @public
	 * @method
	 */
	exports.error = function () {
		_execute('ERROR', arguments);
	};

	/**
	 * Replaces alert
	 *
	 * @param {all} Any data that should be shown in the console statement
	 *
	 * @name Core/Debug.alert
	 * @public
	 * @method
	 */
	exports.alert = function () {
		_execute('ALERT', arguments);
	};

	/**
	 * Debug info for mobile devices.
	 */
	exports.mobile = function () {
		_execute('MOBILE', arguments);
	};

}(jse.core.debug));

/* --------------------------------------------------------------
 engine.js 2016-02-04
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

jse.core.engine = jse.core.engine || {};

/**
 * Main JS Engine Object
 *
 * This object will initialize the page namespaces and collections.
 *
 * @namespace JSE/Core/engine
 */
(function(/** @lends JSE/Core/engine */ exports) {

	'use strict';

	// ------------------------------------------------------------------------
	// PRIVATE FUNCTIONS
	// ------------------------------------------------------------------------

	/**
	 * Initialize the page namespaces.
	 *
	 * This method will search the page HTML for available namespaces.
	 *
	 * @param {array} collections Contains the module collection instances to be included in the namespaces.
	 *
	 * @return {array} Returns an array with the page namespace names.
	 *
	 * @private
	 */
	var _initNamespaces = function(collections) {
		var pageNamespaceNames = [];

		// Use the custom pseudo selector defined at extend.js in order to fetch the available namespaces.
		$(':attr(data-\(.*\)-namespace)').each(function() {
			var $element = $(this);

			$.each($element.data(), function(name, source) {
				if (name.indexOf('Namespace') === -1) {
					return true; // Not a namespace related value.
				}

				name = name.replace('Namespace', ''); // Remove "Namespace" from the data name.

				// Check if the namespace is already defined.
				if (pageNamespaceNames.indexOf(name) > -1) {
					if (window[name].source !== source) {
						jse.core.debug.error('Element with the duplicate namespace name: ', $element[0]);
						throw new Error('The namespace "' + name + '" is already defined. Please select another ' +
							'name for your namespace.');
					}
					return true; // The namespace is already defined, continue loop.
				}

				if (source === '') {
					throw new SyntaxError('Namespace source is empty: ' + name);
				}

				// Create a new namespaces instance in the global scope (the global scope
				// is used for fallback support of old module definitions).
				if (name === 'jse') { // Modify the engine object with Namespace attributes.
					_convertEngineToNamespace(source, collections);
				} else {
					window[name] = new jse.constructors.Namespace(name, source, collections);
				}

				pageNamespaceNames.push(name);
				$element.removeAttr('data-' + name + '-namespace');
			});
		});

		// Throw an error if no namespaces were found.
		if (pageNamespaceNames.length === 0) {
			throw new Error('No module namespaces were found, without namespaces it is not possible to ' +
				'load any modules.');
		}

		// Initialize the namespace instances.
		var deferredCollection = [];
		
		$.each(pageNamespaceNames, function(index, name) {
			var deferred = $.Deferred();
			
			deferredCollection.push(deferred);
			
			window[name]
				.init()
				.done(function() {
					deferred.resolve();
				})
				.fail(function() {
					deferred.reject();
				})
				.always(function() {
					jse.core.debug.info('Namespace promises were resolved: ' , name); 
				});
		});

		// Trigger an event after the engine has initialized all new modules.
		$.when.apply(undefined, deferredCollection).promise().always(function() {
			$('body').trigger('JSENGINE_INIT_FINISHED', []);
		});

		return pageNamespaceNames;
	};

	/**
	 * Convert the "jse" object to a Namespace compatible object.
	 *
	 * In order to support the "jse" namespace name for the core modules placed in the "JSEngine"
	 * directory, we will need to modify the already existing "jse" object so that it can operate
	 * as a namespace without losing its initial attributes.
	 *
	 * @param {string} source Namespace source path for the module files.
	 * @param {array} collections Contain instances to the protoype collection instances.
	 *
	 * @private
	 */
	var _convertEngineToNamespace = function(source, collections) {
		var tmpNamespace = new jse.constructors.Namespace('jse', source, collections);
		jse.name = tmpNamespace.name;
		jse.source = tmpNamespace.source;
		jse.collections = tmpNamespace.collections;
		jse.init = jse.constructors.Namespace.prototype.init;
	};

	// ------------------------------------------------------------------------
	// PUBLIC FUNCTIONS
	// ------------------------------------------------------------------------

	/**
	 * Initialize the engine.
	 *
	 * @param {array} collections Contains the supported module collections prototypes.
	 */
	exports.init = function(collections) {
		// Initialize the page namespaces.
		var pageNamespaceNames = _initNamespaces(collections);

		// Log the page namespaces (for debugging only).
		jse.core.debug.info('Page Namespaces: ' + pageNamespaceNames.join());

		// Update the engine registry.
		jse.core.registry.set('namespaces', pageNamespaceNames);
	};

})(jse.core.engine);

/* --------------------------------------------------------------
 extensions.js 2015-10-13 gm
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2015 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Extend JS Engine
 *
 * Extend the default behaviour of engine components or external plugins before they are loaded.
 *
 * @namespace JSE/Core/extend
 */
(function () {

	'use strict';

	// ------------------------------------------------------------------------
	// NAMESPACE PSEUDO SELECTOR DEFINITION
	// ------------------------------------------------------------------------

	if (typeof $.expr.pseudos.attr === 'undefined') {
		$.expr.pseudos.attr = $.expr.createPseudo(function(selector) {
			var regexp = new RegExp(selector);
			return function(elem) {
				for(var i = 0; i < elem.attributes.length; i++) {
					var attr = elem.attributes[i];
					if(regexp.test(attr.name)) {
						return true;
					}
				}
				return false;
			};
		});
	}


	// ------------------------------------------------------------------------
	// EXTENSION DEFINITION
	// ------------------------------------------------------------------------

	/**
	 * Set jQuery UI datepicker widget defauls.
	 *
	 * @name core/extend.datepicker
	 * @public
	 *
	 * @type {object}
	 */
	$.datepicker.regional.de = {
		dateFormat: 'dd.mm.yy',
		firstDay: 1,
		isRTL: false
	};
	$.datepicker.setDefaults($.datepicker.regional.de);
}());

/* --------------------------------------------------------------
 fallback.js 2015-10-16 gm
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2015 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

jse.core.fallback = jse.core.fallback || {};

/**
 * Fallback Library
 *
 * This library contains a set of deprecated functions that are still present for fallback
 * support. Each function will be removed from the engine after two minor releases.
 *
 * @namespace JSE/Core/fallback
 */
(function (/** @lends JSE/Core/fallback */exports) {

	'use strict';


	$(document).ready(function () {

		// Event listener that performs on every validate
		// validate trigger that isn't handled by the validator
		$('body').on('validator.validate', function (e, d) {
			if (d && d.deferred) {
				d.deferred.resolve();
			}
		});

		// Event listener that performs on every formchanges.check
		// trigger that isn't handled by the form_changes_checker
		$('body').on('formchanges.check', function (e, d) {
			if (d && d.deferred) {
				d.deferred.resolve();
			}
		});

		// Apply touch class to body
		// for touch-devices
		if (jse.core.config.get('hasTouch')) {
			$('body').addClass('has-touch');
		}
	});

	/**
	 * Add a deprecation warning in the console.
	 *
	 * As the JS engine evolves many old features will need to be changed in order to let a
	 * finer and clearer API for the JS Engine core mechanisms. Use this method to create a
	 * deprecation warning for the functions placed within this library.
	 *
	 * @param {string} functionName The deprecated function name.
	 * @param {string} deprecationVersion Deprecation version without the "v".
	 * @param {string} removalVersion Removal version withou the "v"
	 *
	 * @private
	 */
	var _deprecation = function (functionName, deprecationVersion, removalVersion) {
		jse.core.debug.warn('The "' + functionName + '" function is deprecated as of v' + deprecationVersion +
		                    ' and will be removed in v' + removalVersion);
	};

	/**
	 * Setup Widget Attribute
	 *
	 * @param {object} $element Change the widget attribute of an element.
	 *
	 * @deprecated since version 1.2.0 - will be removed in 1.4.0
	 *
	 * @public
	 */
	exports.setupWidgetAttr = function ($element) {
		_deprecation('setupWidgetAttr', '1.2.0', '1.4.0');

		$element
			.filter(':attr(^data-gx-_), :attr(^data-gambio-_)')
			.add($element.find(':attr(^data-gx-_), :attr(^data-gambio-_)'))
			.each(function () {
				var $self = $(this),
					attributes = $self[0].attributes,
					matchedAttribute,
					namespaceName;

				$.each(attributes, function (index, attribute) {
					if (attribute === undefined) {
						return true; // wrong attribute, continue loop
					}

					matchedAttribute = attribute.name.match(/data-(gambio|gx)-_.*/g);

					if (matchedAttribute !== null && matchedAttribute.length > 0) {
						namespaceName = matchedAttribute[0].match(/(gambio|gx)/g)[0];

						$self
							.attr(attribute.name.replace('data-' + namespaceName + '-_',
							                             'data-' + namespaceName + '-'), attribute.value);
					}
				});
			});
	};

	/**
	 * @deprecated since version 1.2.0 - will be removed in 1.4.0
	 * @param {object} data
	 * @param {object} $target
	 * @public
	 */
	exports.fill = function (data, $target) {
		_deprecation('fill', '1.2.0', '1.4.0');

		$.each(data, function (i, v) {
			var $elements = $target
				.find(v.selector)
				.add($target.filter(v.selector));

			$elements.each(function () {
				var $element = $(this);

				switch (v.type) {
					case 'html':
						$element.html(v.value);
						break;
					case 'attribute':
						$element.attr(v.key, v.value);
						break;
					case 'replace':
						if (v.value) {
							$element.replaceWith(v.value);
						} else {
							$element
								.addClass('hidden')
								.empty();
						}
						break;
					default:
						$element.text(v.value);
						break;
				}
			});

		});
	};

	/**
	 * @deprecated since version 1.2.0 - will be removed in 1.4.0
	 * @param url
	 * @param deep
	 * @returns {{}}
	 */
	exports.getUrlParams = function (url, deep) {
		_deprecation('getUrlParams', '1.2.0', '1.4.0');

		url = decodeURIComponent(url || location.href);

		var splitUrl = url.split('?'),
			splitParam = (splitUrl.length > 1) ? splitUrl[1].split('&') : [],
			regex = new RegExp(/\[(.*?)\]/g),
			result = {};

		$.each(splitParam, function (i, v) {
			var keyValue = v.split('='),
				regexResult = regex.exec(keyValue[0]),
				base = null,
				basename = keyValue[0].substring(0, keyValue[0].search('\\[')),
				keys = [],
				lastKey = null;

			if (!deep || regexResult === null) {
				result[keyValue[0]] = keyValue[1].split('#')[0];
			} else {

				result[basename] = result[basename] || [];
				base = result[basename];

				do {
					keys.push(regexResult[1]);
					regexResult = regex.exec(keyValue[0]);
				} while (regexResult !== null);

				$.each(keys, function (i, v) {
					var next = keys[i + 1];
					v = v || '0';

					if (typeof (next) === 'string') {
						base[v] = base[v] || [];
						base = base[v];
					} else {
						base[v] = base[v] || undefined;
						lastKey = v;
					}
				});

				if (lastKey !== null) {
					base[lastKey] = keyValue[1];
				} else {
					base = keyValue[1];
				}
			}

		});

		return result;
	};

	/**
	 * Fallback getData method.
	 *
	 * This method was included in v1.0 of JS Engine and is replaced by the
	 * "jse.libs.form.getData" method.
	 *
	 * @deprecated since version 1.2.0 - will be removed in 1.4.0
	 *
	 * @param {object} $form Selector of the form to be parsed.
	 * @param {string} ignore (optional) jQuery selector string of form elements to be ignored.
	 *
	 * @returns {object} Returns the data of the form as an object.
	 */
	exports.getData = function ($form, ignore) {
		var $elements = $form.find('input, textarea, select'),
			result = {};

		if (ignore) {
			$elements = $elements.filter(':not(' + ignore + ')');
		}

		$elements.each(function () {
			var $self = $(this),
				type = $self.prop('tagName').toLowerCase(),
				name = $self.attr('name'),
				$selected = null;

			type = (type !== 'input') ? type : $self.attr('type').toLowerCase();

			switch (type) {
				case 'radio':
					$form
						.find('input[name="' + name + '"]:checked')
						.val();
					break;
				case 'checkbox':
					if (name.search('\\[') !== -1) {
						if ($self.prop('checked')) {
							name = name.substring(0, name.search('\\['));
							if (typeof result[name] === 'undefined') {
								result[name] = [];
							}
							result[name].push($(this).val());
						}
					} else {
						result[name] = $self.prop('checked');
					}
					break;
				case 'select':
					$selected = $self.find(':selected');
					if ($selected.length > 1) {
						result[name] = [];
						$selected.each(function () {
							result[name].push($(this).val());
						});
					} else {
						result[name] = $selected.val();
					}
					break;
				case 'button':
					break;
				default:
					if (name) {
						result[name] = $self.val();
					}
					break;
			}
		});
		return result;
	};

})(jse.core.fallback);

/* --------------------------------------------------------------
 lang.js 2015-10-13 gm
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2015 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

jse.core.lang = jse.core.lang || {};

/**
 * ## JS Engine Localization Library
 *
 * The global Lang object contains language information that can be easily used in your
 * JavaScript code. The object contains constance translations and dynamic sections that
 * can be loaded and used in different page.
 *
 * #### Important
 * The engine will automatically load translation sections that are present in the
 * `window.JSEngineConfiguration.translations` property upon initialization. For more
 * information look at the "core/initialize" page of documentation reference.
 *
 * ```javascript
 * jse.core.lang.addSection('sectionName', { translationKey: 'translationValue' }); // Add translation section.
 * jse.core.translate('translationKey', 'sectionName'); // Get the translated string.
 * jse.core.getSections(); // returns array with sections e.g. ['admin_buttons', 'general']
 * ```
 *
 * @namespace JSE/Core/lang
 */
(function (exports) {

	'use strict';

	// ------------------------------------------------------------------------
	// VARIABLES
	// ------------------------------------------------------------------------

	/**
	 * Contains various translation sections.
	 *
	 * @type {object}
	 */
	var sections = {};

	// ------------------------------------------------------------------------
	// PUBLIC METHODS
	// ------------------------------------------------------------------------

	/**
	 * Add a translation section.
	 *
	 * @param {string} name Name of the section, used later for accessing translation strings.
	 * @param {object} translations Key - value object containing the translations.
	 *
	 * @throws Exception if "name" or "translations" arguments are invalid.
	 *
	 * @name core/lang.addSection
	 * @public
	 * @method
	 */
	exports.addSection = function (name, translations) {
		if (typeof name !== 'string' || typeof translations !== 'object' || translations === null ||
		    translations.length === 0) {
			throw new Error('window.gx.core.lang.addSection: Invalid arguments provided (name: ' + (typeof name) +
			                ', translations: ' + (typeof translations) + ')');
		}

		sections[name] = translations;
	};

	/**
	 * Get loaded translation sections.
	 *
	 * Useful for asserting present translation sections.
	 *
	 * @return {array} Returns array with the existing sections.
	 *
	 * @name core/lang.getSections
	 * @public
	 * @method
	 */
	exports.getSections = function () {
		var result = [];
		$.each(sections, function (name, content) {
			result.push(name);
		});
		return result;
	};

	/**
	 * Translate string in Javascript code.
	 *
	 * @param {string} phrase Name of the phrase containing the translation.
	 * @param {string} section Section name containing the translation string.
	 *
	 * @returns {string} Returns the translated string.
	 *
	 * @throws Exception if provided arguments are invalid.
	 * @throws Exception if required section does not exist or translation could not be found.
	 *
	 * @name core/lang.translate
	 * @public
	 * @method
	 */
	exports.translate = function (phrase, section) {
		// Validate provided arguments.
		if (typeof phrase !== 'string' || typeof section !== 'string') {
			throw new Error('Invalid arguments provided in translate method (phrase: ' + (typeof phrase) +
			                ', section: ' + (typeof section) + ').');
		}

		// Check if translation exists.
		if (typeof sections[section] === 'undefined' || typeof sections[section][phrase] === 'undefined') {
			jse.core.debug.warn('Could not found requested translation (phrase: ' + phrase + ', section: '
			                    + section + ').');
			return '{' + section + '.' + phrase + '}';
		}

		return sections[section][phrase];
	};

}(jse.core.lang));

/* --------------------------------------------------------------
 module_loader.js 2015-10-16 gm
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2015 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

jse.core.module_loader = jse.core.module_loader || {};

/**
 * ## JS Engine Module Loader
 *
 * This object is an adapter between the engine and RequireJS which is used to
 * load the required files into the client.
 *
 * @namespace JSE/Core/module_loader
 */
(function (/** @lends JSE/Core/module_loader */ exports) {

	'use strict';

	// ------------------------------------------------------------------------
	// PUBLIC METHODS
	// ------------------------------------------------------------------------

	/**
	 * Initialize the module loader.
	 *
	 * Execute this method after the engien config is inialized. It will configure requirejs
	 * so that it will be able to find the project files.
	 */
	exports.init = function () {
		var config = {
			baseUrl: jse.core.config.get('shopUrl'),
			urlArgs: jse.core.config.get('cacheBust') ? 'bust=' + (new Date()).getTime() : '',
			onError: function (error) {
				jse.core.debug.error('RequireJS Error:', error);
			}
		};

		require.config(config);
	};

	/**
	 * Load a module file with the use of requirejs.
	 *
	 * @param {object} $element Selector of the element which has the module definition.
	 * @param {string} name Module name to be loaded. Modules have the same names as their files.
	 * @param {object} collection Current collection instance.
	 *
	 * @return {object} Returns a promise object to be resolved with the module
	 * instance as a parameter.
	 */
	exports.load = function ($element, name, collection) {
		var deferred = $.Deferred();

		try {
			if (name === '') {
				deferred.reject(new Error('Module name cannot be empty.'));
			}

			var baseModuleName = name.replace(/.*\/(.*)/, '$1'); // Name without the parent directory.

			// Try to load the cached instance of the module.
			var cached = collection.cache.modules[baseModuleName];
			if (cached && cached.code === 'function') {
				console.log(collection, collection.namespace);
				deferred.resolve(new jse.constructors.Module($element, baseModuleName, collection));
				return true; // continue loop
			}

			// Try to load the module file from the server.
			var fileExtension = jse.core.config.get('debug') !== 'DEBUG' ? '.min.js' : '.js',
				url = collection.namespace.source + '/' + collection.name + '/' + name + fileExtension;

			require([url], function () {
				if (typeof collection.cache.modules[baseModuleName] === 'undefined') {
					throw new Error('Module "' + name + '" wasn\'t defined correctly. Check the module code for '
					                + 'further troubleshooting.');
				}

				var dependencies = collection.cache.modules[baseModuleName].dependencies.slice(); // use slice for copying the array

				if (dependencies.length === 0) { // no dependencies
					deferred.resolve(new jse.constructors.Module($element, baseModuleName, collection));
					return true; // continue loop
				}

				// Load the dependencies first.
				$.each(dependencies, function (index, dependency) {
					if (dependency.indexOf('http') === -1) { // Then convert the relative path to JSEngine/libs directory.
						dependencies[index] = jse.core.config.get('engineUrl') + '/libs/' + dependency + fileExtension;
					} else if (dependency.indexOf('.js') === -1) { // Then add the dynamic file extension to the URL.
						dependencies[index] += fileExtension;
					}
				});

				require(dependencies, function () {
					deferred.resolve(new jse.constructors.Module($element, baseModuleName, collection));
				});
			});
		} catch (exception) {
			deferred.reject(exception);
		}

		return deferred.promise();
	};

})(jse.core.module_loader);

/* --------------------------------------------------------------
 polyfills.js 2015-10-28 gm
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2015 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Polyfills for cross-browser compatibility.
 *
 * @namespace JSE/Core/polyfills
 */
(function () {

	'use strict';

	if (!Array.prototype.indexOf) {
		Array.prototype.indexOf = function (searchElement, fromIndex) {
			var k;
			if (this == null) {
				throw new TypeError('"this" is null or not defined');
			}

			var O = Object(this);
			var len = O.length >>> 0;

			if (len === 0) {
				return -1;
			}

			var n = +fromIndex || 0;

			if (Math.abs(n) === Infinity) {
				n = 0;
			}

			if (n >= len) {
				return -1;
			}

			k = Math.max(n >= 0 ? n : len - Math.abs(n), 0);

			while (k < len) {
				var kValue;
				if (k in O && O[k] === searchElement) {
					return k;
				}
				k++;
			}
			return -1;
		};
	}

	// Internet Explorer does not support the origin property of the window.location object.
	// @link http://tosbourn.com/a-fix-for-window-location-origin-in-internet-explorer
	if (!window.location.origin) {
		window.location.origin = window.location.protocol + '//' +
		                         window.location.hostname + (window.location.port ? ':' + window.location.port : '');
	}

	// Date.now method polyfill
	// https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Date/now
	if (!Date.now) {
		Date.now = function now() {
			return new Date().getTime();
		};
	}
})();



/* --------------------------------------------------------------
 registry.js 2015-10-13 gm
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2015 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

jse.core.registry = jse.core.registry || {};

/**
 * ## JS Engine Registry
 *
 * This object contains string data that other sections of the engine need in order to
 * operate correctly.
 *
 * @namespace JSE/Core/registry
 */

(function (/** @lends Admin/Libs/registry */ exports) {

	'use strict';

	var registry = [];

	/**
	 * Set a value in the registry.
	 *
	 * @param {string} name Contains the name of the entry to be added.
	 * @param {string} value The value to be written in the registry.
	 *
	 * @public
	 */
	exports.set = function (name, value) {
		// If a registry entry with the same name exists already the following console warning will
		// inform developers that they are overwriting an existing value, something useful when debugging.
		if (typeof registry[name] !== 'undefined') {
			jse.core.debug.warn('The registry value with the name "' + name + '" will be overwritten.');
		}

		registry[name] = value;
	};

	/**
	 * Get a value from the registry.
	 *
	 * @param {string} name The name of the entry value to be returned.
	 *
	 * @returns {*} Returns the value that matches the name.
	 */
	exports.get = function (name) {
		return registry[name];
	};

	/**
	 * Check the current content of the registry object.
	 *
	 * This method is only available when the engine environment is turned into
	 * development.
	 *
	 * @public
	 */
	exports.debug = function () {
		if (jse.core.config.get('environment') === 'development') {
			jse.core.debug.log('Registry Object:', registry);
		} else {
			throw new Error('This function is not allowed in a production environment.');
		}
	};

})(jse.core.registry);

//# sourceMappingURL=data:application/json;base64,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
