/* --------------------------------------------------------------
 emails_table.js 2015-07-21 gm
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2015 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Emails Table Controller
 *
 * This controller will handle the main table operations of the admin/emails page.
 *
 * @module Controllers/emails_table
 * @version 0.1
 */
window.gambio.controllers.register('emails_table', 0.1, [ 'emails' ], /** @lends module:Controllers/emails_table */ function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLE DEFINITION
    // ------------------------------------------------------------------------

    var
        /**
         * Module Reference
         *
         * @type {object}
         */
        $this = $(this),

        /**
         * Toolbar Selector 
         * 
         * @type {object}
         */
        $toolbar = $('#emails-toolbar'),

        /**
         * Modal Selector
         * 
         * @type {object}
         */
        $modal = $('#emails-modal'),
        
        /**
         * Default Module Options
         *
         * @type {object}
         */
        defaults = {
            emailsTableActions: function() {
                return '<button class="send-email btn btn-icon remove-margin" title="' + Lang.translate('send', 'buttons') + '">'
                    + '<img src="' + Config.shopUrl() + '/admin/images/export_schemes/icon_export.png" />'
                    + '</button>' 
                    + '<button class="forward-email btn btn-icon remove-margin" title="' + Lang.translate('forward', 'buttons') + '">'
                    + '<img src="' + Config.shopUrl() + '/admin/images/export_schemes/icon_copy.png" />'
                    + '</button>'
                    + '<button class="delete-email btn btn-icon remove-margin" title="' + Lang.translate('delete', 'buttons') + '">'
                    + '<img src="' + Config.shopUrl() + '/admin/images/export_schemes/icon_delete.png" />'
                    + '</button>'
                    + '<button class="preview-email btn btn-icon remove-margin" title="' + Lang.translate('preview', 'buttons') + '">'
                    + '<img src="' + Config.shopUrl() + '/admin/images/export_schemes/icon_preview.png" />'
                    + '</button>';
            },

            convertPendingToString: function(data, type, row, meta) {
                return (data === true) ? Lang.translate('email_pending', 'emails') : Lang.translate('email_sent', 'emails');
            }
        },

        /**
         * Final Module Options
         *
         * @type {object}
         */
        options = $.extend(true, {}, defaults, data),

        /**
         * Meta Object
         *
         * @type {object}
         */
        meta = {};


    // ------------------------------------------------------------------------
    // EVENT HANDLERS
    // ------------------------------------------------------------------------

    /**
     * Toggle row selection for main page table.
     *
     * @param {object} event Contains event information.
     */
    var _onSelectAllRows = function(event) {
        var isChecked = $(this).prop('checked');
        $this.find('input[type=checkbox]').prop('checked', isChecked);
    };

    /**
     * Will send the email to its contacts (even if its status is "sent").
     *
     * @param {object} event Contains event information.
     */
    var _onSendEmail = function (event) {
        var $row = $(this).parents('tr');

        Modal.message({
            title: Lang.translate('send', 'buttons'),
            content: Lang.translate('prompt_send_email', 'emails'),
            buttons: [
                {
                    text: Lang.translate('yes', 'lightbox_buttons'),
                    click: function() {
                        var email = $row.data();
                        window.gambio.libs.emails.sendCollection([ email ])
                            .done(function(response) {
                                $this.DataTable().ajax.reload();
                                window.gambio.libs.emails.getAttachmentsSize($('#attachments-size'));
                            })
                            .fail(function(response) {
                                Modal.message({
                                    title: Lang.translate('error', 'messages'),
                                    content: response.message
                                }) 
                            });
                        $(this).dialog('close');
                    }
                },
                {
                    text: Lang.translate('no', 'lightbox_buttons'),
                    click: function() {
                        $(this).dialog('close');
                    }
                }
            ]
        });
    };

    /**
     * Display modal with email information but without contacts.
     *
     * The user will be able to set new contacts and send the email (kind of "duplicate" method).
     *
     * @param {object} event Contains event information.
     */
    var _onForwardEmail = function(event) {
        var email = $(this).parents('tr').data();

        window.gambio.libs.emails.resetModal($modal);
        window.gambio.libs.emails.loadEmailOnModal(email, $modal);

        // Clear contact fields but let the rest of the data untouched.
        $modal.find('#email-id').val('');
        $modal.find('#sender-email, #sender-name').val('');
        $modal.find('#reply-to-email, #reply-to-name').val('');
        $modal.find('#recipient-email, #recipient-name').val('');
        $modal.find('#contacts-table').DataTable().clear().draw();

        $modal.dialog({
            title: Lang.translate('forward', 'buttons'),
            width: 1000,
            height: 750,
            modal: true,
            position: 'top+15',
            dialogClass: 'gx-container',
            closeOnEscape: false,
            buttons: window.gambio.libs.emails.getDefaultModalButtons($modal, $this),
            open: window.gambio.libs.emails.colorizeButtonsForEditMode
        });
    };

    /**
     * Delete selected row email.
     *
     * @param {object} event Contains event information.
     */
    var _onDeleteEmail = function (event) {
        var $row = $(this).parents('tr'),
            email = $row.data();

        Modal.message({
            title: Lang.translate('delete', 'buttons'),
            content: Lang.translate('prompt_delete_email', 'emails'),
            buttons: [
                {
                    text: Lang.translate('yes', 'lightbox_buttons'),
                    click: function() {
                        window.gambio.libs.emails.deleteCollection([ email ])
                            .done(function(response) {
                                $this.DataTable().ajax.reload();
                                window.gambio.libs.emails.getAttachmentsSize($('#attachments-size'));
                            })
                            .fail(function(response) {
                                Modal.message({
                                    title: Lang.translate('error', 'messages'),
                                    content: response.message
                                })
                            });
                        $(this).dialog('close');
                    }
                },
                {
                    text: Lang.translate('no', 'lightbox_buttons'),
                    click: function() {
                        $(this).dialog('close');
                    }
                }
            ]
        });
    };

    /**
     * Display modal with email information
     *
     * The user can select an action to perform upon the previewed email (Send, Forward,
     * Delete, Close).
     *
     * @param  {object} event Contains event information.
     */
    var _onPreviewEmail = function (event) {
        var email = $(this).parents('tr').data();

        window.gambio.libs.emails.resetModal($modal);
        window.gambio.libs.emails.loadEmailOnModal(email, $modal);

        $modal.dialog({
            title: Lang.translate('preview', 'buttons'),
            width: 1000,
            height: 750,
            modal: true,
            position: 'top+15',
            dialogClass: 'gx-container',
            closeOnEscape: false,
            buttons: window.gambio.libs.emails.getPreviewModalButtons($modal, $this),
            open: window.gambio.libs.emails.colorizeButtonsForPreviewMode
        });
    }; 


    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------
    
    /**
     * Initialize method of the module, called by the engine. 
     * 
     * The emails table operates with server processing because it is much faster and efficient than preparing 
     * and sending all the records in every AJAX request. Check the Emails/DataTable controller method for 
     * requested data and the following link for more info about server processing in DataTables.
     * 
     * {@link http://www.datatables.net/manual/server-side}
     */
    meta.init = function () {        
        // Create a DataTable instance for the email records.
        DataTable.create($this, {
            processing: true,
            serverSide: true,
            dom: 'rtip', 
            autoWidth: false,
            language: (Config.languageCode() === 'de') ? DataTable.getGermanTranslation() : null,
            ajax:  {
                url: Config.shopUrl() + '/admin/admin.php?do=Emails/DataTable',
                type: 'POST'
            },
            order: [[2, 'desc']],
            pageLength: 50,
            columns: [
                { data: null, orderable: false, defaultContent: '<input type="checkbox" />', width: '2%', 
                        className: 'dt-head-center dt-body-center' },
                { data: 'row_count', orderable: false, width: '3%', className: 'dt-head-center dt-body-center' },
                { data: 'creation_date', width: '12%'},
                { data: 'sent_date', width: '12%' },
                { data: 'sender', width: '12%' },
                { data: 'recipient', width: '12%' },
                { data: 'subject', width: '24%' },
                { data: 'is_pending', width: '8%', render: options.convertPendingToString },
                { data: null, orderable: false, defaultContent: '', render: options.emailsTableActions, width: '15%' }
            ]
        });
        
        // Add table error handler.
        DataTable.error($this, function(event, settings, techNote, message) {
            Modal.message({
                title: 'DataTables ' + Lang.translate('error', 'messages'),
                content: message
            });
        });
        
        // Add ajax error handler.
        DataTable.ajaxComplete($this, function(event, settings, json) {
            if (json.exception === true) {
                Dbg.error('DataTables Processing Error', $this.get(0),  json);
                Modal.message({
                    title: 'AJAX ' + Lang.translate('error', 'messages'),
                    content: json.message
                });
            }
        });
        
        // Bind event handlers of the emails table. 
        $this
            .on('click', '#select-all-rows', _onSelectAllRows)
            .on('click', '.send-email', _onSendEmail)
            .on('click', '.forward-email', _onForwardEmail)
            .on('click', '.delete-email', _onDeleteEmail)
            .on('click', '.preview-email', _onPreviewEmail);
        
        meta._initFinished();
    };

    // Return data to module engine.
    return meta;
});